#!/bin/sh

# This script can be used to analyze the deadlock information reported
# in the errorlog of a Sybase ASE.
# The file errorlog should be present. When you have multiple errorlog
# files (from one server) you can append them into one file.
# The errorlog file from a server should be analyzed on the same server.
#
# Documentation: see www.petersap.nl
#
# This script may be copied or redistributed
# Written by Peter Sap
#
# All disclaimers apply
#
# Version history:
# Oct-01-2004	- 1.0	Initial version
# Nov-15-2004	- 1.0.1	Performance improvement in analysis phase
# Sep-02-2005	- 1.1	Retrieve and use character-set for bcp
# Dec-18-2006	- 1.2	Compatible with ASE15, improved error trapping
# Aug-16-2007	- 1.3	Use printf i.s.o. echo to handle quotes better on SunOs

if [ $# -lt 2 ]
then
  echo "Usage $0 <servername> <username> <password>"
  exit 1
fi

SERVER=$1
USERNAME=$2
PASSWORD=$3

echo	"...Checking requirements"

if	[ ! -f errorlog ]
then
	echo "ERROR: No file 'errorlog' found."
	exit 1
fi

type isql > /tmp/$$.check

COUNT=`grep "not found" /tmp/$$.check | wc -l`
if [ ${COUNT} -gt 0 ]
then
	echo "ERROR: isql not in path."
	rm -f /tmp/$$.*

	exit 1
fi

type bcp > /tmp/$$.check

COUNT=`grep "not found" /tmp/$$.check | wc -l`
if [ ${COUNT} -gt 0 ]
then
	echo "ERROR: bcp not in path."
	rm -f /tmp/$$.*

	exit 1
fi

isql -U${USERNAME} -P${PASSWORD} -S${SERVER} <<EOF > /tmp/$$.check
sp_activeroles
go
EOF

COUNT=`grep CT-LIBRARY /tmp/$$.check | wc -l`
if [ ${COUNT} -gt 0 ]
then
	cat /tmp/$$.check
	rm -f /tmp/$$.*

	exit 1
fi

COUNT=`grep sa_role /tmp/$$.check | wc -l`
if [ ${COUNT} -eq 0 ]
then
	echo "ERROR: You need to have sa_role to use this tool."
	rm -f /tmp/$$.*

	exit 1
fi

COUNT=`grep sybase_ts_role /tmp/$$.check | wc -l`
if [ ${COUNT} -eq 0 ]
then
	echo "ERROR: You need to have sybase_ts_role to use this tool."
	rm -f /tmp/$$.*

	exit 1
fi

isql -U${USERNAME} -P${PASSWORD} -S${SERVER} -b <<EOF > /tmp/$$.charset
set	nocount on
go
select	ch.name
	from	master..syscharsets ch,
		master..sysconfigures co
	where	co.config	= 131
	and	co.value	= ch.id
	and	ch.type		< 2000
go
if	@@rowcount != 1
	print	"Msg -- could not determine characterset"
go
EOF

COUNT=`grep Msg /tmp/$$.charset | wc -l`
if [ ${COUNT} -gt 0 ]
then
  cat /tmp/$$.charset
  rm -f /tmp/$$.*
  exit 1
fi

CHARSET=`cat /tmp/$$.charset | awk '{ print $1 }'`
echo	"...Server is using character set ${CHARSET}"

isql -U${USERNAME} -P${PASSWORD} -S${SERVER} -b <<EOF > /tmp/$$.version
set	nocount on
go
select	@@version_as_integer
go
EOF

COUNT=`grep Msg /tmp/$$.version | wc -l`
if [ ${COUNT} -gt 0 ]
then
	VERSION=11000
	echo "...Detected ASE version pre 12.x"
else
	VERSION=`cat /tmp/$$.version | awk '{ print $1 }'`
	echo "...Detected ASE version ${VERSION}"
fi

echo	"...Starting deadlock analysis"
isql -U${USERNAME} -P${PASSWORD} -S${SERVER} <<EOF > /tmp/$$.start
use	tempdb
go
--
-- Drop some tables when they already exist
--
if	object_id("errorlog")	!= null
	drop	table	errorlog
go
if	object_id("pages")	!= null
	drop	table	pages
go
if	object_id("pageinfo")	!= null
	drop	table	pageinfo
go
if	object_id("tables")	!= null
	drop	table	tables
go
if	object_id("dbccoutput") != null
	drop	table	dbccoutput
go
--
-- Create a table to store the errorlog
--
create	table errorlog(
line	varchar(512)	null)
go
EOF

COUNT=`grep Msg /tmp/$$.start | wc -l`
if [ ${COUNT} -gt 0 ]
then
  cat /tmp/$$.start
  rm -f /tmp/$$.*
  exit 1
fi

rm -f /tmp/$$.analyse
touch /tmp/$$.analyse

#
# load errorlog file into the database
#
echo	"...bcp in of errorlog"

bcp tempdb..errorlog in errorlog -U${USERNAME} -P${PASSWORD} -S${SERVER} -c -b1000 -J${CHARSET} >> /tmp/$$.analyse

COUNT=`grep Msg /tmp/$$.analyse | wc -l`
if [ ${COUNT} -gt 0 ]
then
  cat /tmp/$$.analyse
  rm -f /tmp/$$.*
  exit 1
fi

echo	"...creating dbcc page() script"

isql -U${USERNAME} -P${PASSWORD} -S${SERVER} -b <<EOF > /tmp/$$.dbccpage.sql
use	tempdb
go
set	nocount on
go

create	table	pages(
	pagenr		int		not null,
	dbid		smallint	not null,
	tablename	varchar(30)	not null,
	page_or_row	char(1)		not null)
go

create	table	tables(
	name		varchar(30)	not null,
	dbid		smallint	not null)
go

--
-- insert all pagenumbers and their dbid's for Deadlocks into a table
--

insert	into	pages
	select	pagenr	= convert(int,substring(line,charindex("on page",line)+8,charindex(" ",substring(line,charindex("on page",line)+8,512))-1)),
		dbid	= case
			when ${VERSION} >= 15000
			then db_id(substring(line,charindex("in database",line)+13,charindex(" ",substring(line,charindex("in database",line)+13,512))-2))
			else convert(int,substring(line,charindex("in database",line)+12,charindex(" ",substring(line,charindex("in database",line)+12,512))-1))
			end,
		tablename	= substring(line,charindex("of the",line)+8,charindex(" ",substring(line,charindex("of the",line)+8,512))-2),
		"P"
		from	errorlog
		where	line	like "Deadlock%on page%in database%"
go

insert	into	pages
	select	pagenr	= convert(int,substring(line,charindex(" page ",line)+6,charindex(" ",substring(line,charindex(" page ",line)+6,512))-1)),
		dbid	= case
			when ${VERSION} >= 15000
			then db_id(substring(line,charindex("in database",line)+13,charindex(" ",substring(line,charindex("in database",line)+13,512))-2))
			else convert(int,substring(line,charindex("in database",line)+12,charindex(" ",substring(line,charindex("in database",line)+12,512))-1))
			end,
		tablename	= substring(line,charindex("of the",line)+8,charindex(" ",substring(line,charindex("of the",line)+8,512))-2),
		"R"
		from	errorlog
		where	line	like "Deadlock%lock on row%in database%"
go

insert	into	tables
	select	name	= substring(line,charindex("lock on the",line)+13,charindex(" ",substring(line,charindex("lock on the",line)+13,512))-2),
		dbid	= case
			when ${VERSION} >= 15000
			then db_id(substring(line,charindex("in database",line)+13,charindex(" ",substring(line,charindex("in database",line)+13,512))-2))
			else convert(int,substring(line,charindex("in database",line)+12,charindex(" ",substring(line,charindex("in database",line)+12,512))-1))
			end
		from	errorlog
		where	line	like "Deadlock%was waiting for a%lock on the%table%"
go

-- 
-- scroll through the distinct values of pagenr and dbid to create a
-- dbcc page command
--
declare	lees cursor for
	select	distinct
		pagenr,
		dbid
		from	pages
		order	by dbid,pagenr
go

declare	@pagenr	int,
	@dbid	int

print	"dbcc traceon(3604)"
print	"go"

open	lees

fetch	lees into
	@pagenr,
	@dbid

while	@@sqlstatus = 0
begin
	print	"dbcc page(%1!,%2!)",
		@dbid,
		@pagenr
	print	"go"

	fetch	lees into
		@pagenr,
		@dbid
end

close	lees
go

deallocate cursor lees
go

--
-- create a table that we can use to store the output of the dbcc page
--
create	table dbccoutput(
line	varchar(512)	null,
seq	numeric(8,0)	identity not null)
go
EOF

COUNT=`grep Msg /tmp/$$.dbccpage.sql | wc -l`
if [ ${COUNT} -gt 0 ]
then
  cat /tmp/$$.dbccpage.sql
  rm -f /tmp/$$.*
  exit 1
fi

echo	"...executing dbcc page() script"

isql -U${USERNAME} -P${PASSWORD} -S${SERVER} -b -w1024 < /tmp/$$.dbccpage.sql > /tmp/$$.dbccpage.out

COUNT=`grep Msg /tmp/$$.dbccpage.out | wc -l`
if [ ${COUNT} -gt 0 ]
then
  cat /tmp/$$.dbccpage.out
  rm -f /tmp/$$.*
  exit 1
fi

echo	"...storing results"

echo	"10.0" > /tmp/$$.dbccoutput.fmt
echo	"1" >> /tmp/$$.dbccoutput.fmt
printf "%s\"%s\"%s\n" "1 SYBCHAR 0 512 " "\n" " 1 line" >> /tmp/$$.dbccoutput.fmt

bcp tempdb..dbccoutput in /tmp/$$.dbccpage.out -U${USERNAME} -P${PASSWORD} -S${SERVER} -f/tmp/$$.dbccoutput.fmt -b1000 -J${CHARSET} > /tmp/$$.store

COUNT=`grep Msg /tmp/$$.store | wc -l`
if [ ${COUNT} -gt 0 ]
then
  cat /tmp/$$.store
  rm -f /tmp/$$.*
  exit 1
fi

echo	"...starting analysis"

isql -U${USERNAME} -P${PASSWORD} -S${SERVER} <<EOF
use	tempdb
go
set	nocount on
go
select	seq,
	seqPlus	= seq + 1
	into	#pagelines
	from	tempdb..dbccoutput
	where	line	like "pageno=%"
go
create	unique index i1
	on #pagelines(seq)
go
create	unique index i1
	on tempdb..dbccoutput(seq)
go
--
-- create a table for query purposes
--
create	table	pageinfo(
pagenr		int		not null,
objid		int		null,
indid		int		not null,
tablename	varchar(255)	null)
go
--
-- build query table
--
insert	into	pageinfo
	select	pagenr	= convert(int,substring(d1.line,8,charindex(" ",d1.line)-8)),
		objid	=
			case	when ${VERSION} >= 15000
				then	null
				else	convert(int,substring(d1.line,charindex("objid=",d1.line)+6,charindex("timestamp=",d1.line)-charindex("objid=",d1.line)-7))
			end,
		indid	= convert(int,substring(d2.line,charindex("indid=",d2.line)+6,charindex("freeoff=",d2.line)-charindex("indid=",d2.line)-7)),
		tablename	= null
		from	tempdb..dbccoutput d1,
			tempdb..dbccoutput d2,
			#pagelines p
		where	p.seq		= d1.seq
		and	d2.seq		= p.seqPlus
go
--
-- Delete the rows where the tablename from the pageinfo table differs from
-- the errorlog. This can happen when a page has been deallocated and is now
-- in use by a different object.
-- This check is skipped for ASE 15+ since dbcc page does not provide the
-- objid anymore.
--
if	${VERSION}	< 15000
	delete	pageinfo
		from	pageinfo pi
		where	exists(
			select	1
				from	pages p
				where	p.pagenr	= pi.pagenr
				and	p.tablename	!= object_name(pi.objid,p.dbid))
go

--
-- Store the tablename for easy queries.
update	pageinfo
	set	tablename	= p.tablename
	from	pageinfo pi,
		pages p
	where	pi.pagenr	= p.pagenr
go
set	nocount off
go
--
-- Now we are finished.
-- Three sample queries are given here to demonstrate the information we know
-- Show a list of deadlocks on table level, sorted by number of deadlocks
--
select	convert(varchar(59),db_name(t.dbid)
		+ ".."
		+ t.name) as "Object",
	count(*) as "Deadlocks on table"
	from	tables t
	group	by convert(varchar(59),db_name(t.dbid) + ".." + t.name)
	order	by 2
go
--
-- Show a list of deadlocks per object, sorted by number of deadlocks
--
select	convert(varchar(49),db_name(p.dbid)
		+ ".."
		+ pi.tablename) as "Object",
	p.page_or_row as "Page/Row",
	count(*) as "Deadlocks per table"
	from	pageinfo pi,
		pages p
	where	p.pagenr	= pi.pagenr
	group	by convert(varchar(49),db_name(p.dbid) + ".." + pi.tablename),p.page_or_row
	order	by 3
go
--
-- Show a list of deadlocks per object and indid, sorted by object name
-- This query is usefull to determine a possible change in locking schema
--
select	convert(varchar(46),db_name(p.dbid)
		+ ".."
		+ pi.tablename) as "Object",
	indid,
	count(*) as "Deadlocks per index"
	from	pageinfo pi,
		pages p
	where	p.pagenr	= pi.pagenr
	group	by convert(varchar(46),db_name(p.dbid) + ".." + pi.tablename), indid
	order	by 1,2
go
--
-- Cleanup
--
/*
drop	table	pages,
		pageinfo,
		dbccoutput,
		errorlog,
		tables
*/
go
EOF

# 
# More cleanup
#

rm -f /tmp/$$.*

echo	"Done."
echo	""
echo	"When you believe that the generated output is incorrect, please send a bug"
echo	"report to peter@petersap.nl. Attach the errorlog to your mail."
