# Install thresholds in databases that will be activated when free space
# in the logsegment drops below a certain percentage.

# First, the existing thresholds associated with the stored procedure
# sp_tranlogusage are dropped. Then the thresholds are added at certain
# points. Use the PERCENTAGE_LIST variable for this.
# Thresholds must be between a certain distance from each other, this
# is taken into account.

# May be copied and redistributed freely with the inclusion of this:
# Written by Peter Sap (www.petersap.nl)

# The PERCENTAGE_LIST variable sets the points where a threshold should
# be installed. Use a comma separated list.
# Example: 40,20 means that the threshold is set at 40 and 20 % of free
# space left.

PERCENTAGE_LIST=40,30,20,10
export PERCENTAGE_LIST

#
# Validate parameters.
# First two are required (server and username). Password is optional since
# it can be null (not recommended).
# Specify a blank password as ""
#
if [ $# -lt 3 ]
then
  echo "Usage: $0 <servername> <username> <password>"
  exit 1
fi
 
#
# Make a list of all databases on the server
# Skip the model database
#
isql -U$2 -P$3 -S$1 -b<<EOF > /tmp/dblist
set	nocount on
go
select	name
	from	master.dbo.sysdatabases
	where	name	!= "model"
	order	by name
go
EOF

#
# Drop and recreate new thresholds
#
for DATABASE in `cat /tmp/dblist`
do
isql -U$2 -P$3 -S$1 -b<<EOF
set	nocount on
set	chained off
go
use	${DATABASE}
go
print	"Database: ${DATABASE}"
go

-- declare a cursor to go through all the thresholds associated with the
-- transaction log and in use by the sp_tranlogusage stored procedure
declare	thres cursor for
	select	free_space
		from	systhresholds
		where	segment		= 2
		and	proc_name	= "sp_tranlogusage"
go

declare	@free_space	int,
	@error		int

open	thres

fetch	thres into
	@free_space

while	@@sqlstatus = 0
begin
	-- drop the threshold
	exec	@error	= sp_dropthreshold	"${DATABASE}",
						"logsegment",
						@free_space

	if	@error	!= 0
		break

	fetch	thres into
		@free_space
end

close	thres
go

deallocate cursor thres
go

declare	@size	int,
	@perc	int,
	@hyst	int,
	@percs	varchar(255),
	@pos	tinyint,
	@error	int

select	@percs	= "${PERCENTAGE_LIST}",
	@pos	= 0,
	@hyst	= (2 * @@thresh_hysteresis) - 1

-- Calculate the total size within the database that is available for the
-- transaction log
select	@size	= sum(size)
	from	master..sysusages
	where	dbid		= db_id("${DATABASE}")
	and	segmap & 4	= 4

while	1 = 1
begin
	-- Get an element from the predefined PERCENTAGE_LIST variable
	select	@pos	= charindex(",",@percs),
		@perc	= 0

	if	@pos	> 0
		select	@perc	= convert(tinyint,substring(@percs,1,@pos - 1)),
			@percs	= substring(@percs,@pos + 1,255)
	else
		if	@percs	= ""
			break
		else
			select	@perc	= convert(tinyint,@percs),
				@percs	= ""

	-- Round the percentage to zero decimals
	select	@perc	= round((@size / 100.0) * @perc,0)

	-- Can we safely install another threshold at the required position?
	-- This can not be done when another threshold is close to it
	if	not exists(
		select	1
			from	systhresholds
			where	segment		= 2 -- logsegment
			and	free_space	between (@perc - @hyst) and (@perc + @hyst))
	begin
		-- Check for existence of the stored procedure
		if	not exists(
			select	1
				from	sybsystemprocs.dbo.sysobjects
				where	type	= "P"
				and	name	= "sp_tranlogusage")
		begin
			print	"The sp_tranlogusage stored procedure is not installed, do this first."

			break
		end

		exec	@error	= sp_addthreshold	"${DATABASE}",
							"logsegment",
							@perc,
							"sp_tranlogusage"

		if	@error	!= 0
			break
	end
end
go
EOF
done

rm -f /tmp/dblist
